# Live Domain 500 Error Analysis & Fix

## Issue Summary
**Error**: Undefined variable `$web_config` on live domain  
**Status**: CRITICAL - Prevents entire application from working  
**Root Cause**: Silent exception in AppServiceProvider  

---

## Problem Details

### Error Stack Trace
```
Undefined variable $web_config at /home/wgtexsuc/live.wgtesthub.com/resources/themes/default/web-views/home.blade.php:3
```

### Why It's Happening
The `AppServiceProvider.php` boot method has a critical flaw:

**Location**: `app/Providers/AppServiceProvider.php` (Lines 77-230)

```php
try {
    if (Schema::hasTable('business_settings')) {
        // ... web_config is defined here (line 90)
        $web_config = [ ... ];
        
        // ... more setup code
        
        // View::share() only called inside specific IF condition (line 225)
        View::share(['web_config' => $web_config, 'language' => $language]);
    }
} catch (\Exception $exception) {
    // PROBLEM: Exception is silently swallowed!
    // $web_config is NEVER shared with views if ANY exception occurs
}
```

### Common Triggers on Live Server
1. **Database Connection Issues**
   - Wrong host/port
   - Database doesn't exist
   - User permissions issue
   - Network/firewall blocking
   
2. **Missing Database Tables**
   - Migration not run: `business_settings` table missing
   - Migrations failed during deployment
   
3. **Configuration Issues**
   - APP_DEBUG=false (hides actual error)
   - Wrong database credentials
   - Missing .env file or incorrect values
   
4. **Query Execution Failure**
   - Queries in lines 80-224 fail
   - Cache retrieval fails
   - Schema checks fail

---

## Configuration Differences: Local vs Live

### Your Local .env
```
APP_DEBUG=false (but Laravel debug shows errors locally anyway)
DB_HOST=localhost
DB_DATABASE=multivendor
DB_USERNAME=root
DB_PASSWORD=root
```

### Live .env (Typical cPanel/Shared Hosting)
Should have:
```
DB_HOST=localhost  (or your_domain.com)
DB_PORT=3306
DB_DATABASE=live_database_name
DB_USERNAME=live_db_user
DB_PASSWORD=live_db_password
```

---

## Deployment Checklist

### Before Deploying to Live

1. **Database**
   - [ ] Create fresh database on live server
   - [ ] Create database user with proper permissions
   - [ ] Test connection from server
   ```bash
   php artisan tinker
   > DB::connection()->getPdo();  // Should not throw error
   ```

2. **Environment Configuration**
   - [ ] Copy `.env.example` to `.env`
   - [ ] Update all values for live domain:
   ```
   APP_NAME=YourSiteName
   APP_ENV=production
   APP_DEBUG=false
   APP_KEY=base64:YOUR_ACTUAL_KEY
   APP_URL=https://yourdomain.com
   
   DB_CONNECTION=mysql
   DB_HOST=localhost
   DB_PORT=3306
   DB_DATABASE=live_db_name
   DB_USERNAME=live_user
   DB_PASSWORD=live_pass
   ```

3. **Migrations & Seeds**
   - [ ] Run migrations: `php artisan migrate --force`
   - [ ] Seed database: `php artisan db:seed --force`
   - [ ] Verify `business_settings` table exists: `SELECT * FROM business_settings LIMIT 1;`

4. **File Permissions**
   - [ ] Set storage directory writable: `chmod -R 775 storage/`
   - [ ] Set bootstrap/cache writable: `chmod -R 775 bootstrap/cache/`

5. **Cache & Optimization**
   - [ ] Clear config cache: `php artisan config:cache`
   - [ ] Clear view cache: `php artisan view:clear`
   - [ ] Optimize autoloader: `php artisan optimize`

---

## Solution Approach

### Option 1: Emergency Fix (Quick)
Add error logging to see actual exception:

**File**: `app/Providers/AppServiceProvider.php`  
**Location**: Line 232 (in catch block)

```php
} catch (\Exception $exception) {
    \Log::error('AppServiceProvider error: ' . $exception->getMessage());
    \Log::error($exception->getTraceAsString());
    // This will log to storage/logs/laravel.log
}
```

Then check live logs: `tail -f /home/username/public_html/storage/logs/laravel.log`

### Option 2: Proper Fix (Recommended)
Ensure `web_config` is always shared, even if partial data:

Move the `View::share()` outside the if condition to ensure it's always called with at least basic config.

### Option 3: Database Schema First
Ensure `business_settings` table exists before all operations:

```bash
# On live server, via SSH:
php artisan migrate --force
php artisan db:seed --force
```

---

## Subdomain vs Public HTML Deployment

### Deploying to Subdomain (Recommended)
```
Structure: subdomain.yourdomain.com
Public Root: /home/username/public_html/subdomain/public
```

**Steps**:
1. Create subdomain in cPanel
2. Upload all files to `/public_html/subdomain/` 
3. Ensure `.env` is in `/public_html/subdomain/.env` (NOT in public/)
4. Set document root to `/public_html/subdomain/public`
5. Run migrations there

**Advantages**:
- Isolated from main domain
- Easy to test
- Can delete easily without affecting main site

### Deploying to Public HTML Root
```
Structure: yourdomain.com
Public Root: /home/username/public_html/public
```

**Caution**:
- This is the PRODUCTION domain
- Any errors affect main website
- Make sure everything is tested first

---

## Common Live Domain Issues & Fixes

| Issue | Cause | Fix |
|-------|-------|-----|
| `Undefined variable $web_config` | DB connection fails | Check DB credentials in .env |
| `SQLSTATE[HY000]` | Database not found | Run `php artisan migrate` |
| `404 Not Found` on all pages | Wrong document root | Point to `/public` folder |
| `Blank white page` | PHP error, debug=false | Check `storage/logs/laravel.log` |
| `Permission denied` on storage | Wrong ownership | `chown -R username:username storage/` |

---

## Testing on Live (After Fix)

1. **Test homepage**: `https://yourdomain.com/`
   - Should display content without 500 error
   - Images should load
   - "Delivery" terminology should show
   
2. **Check logs**: `tail -50 storage/logs/laravel.log`
   - Should show normal requests, no errors
   
3. **Test core pages**:
   - Products page
   - Cart
   - Checkout
   - User profile

---

## Immediate Next Steps

1. **SSH into Live Server** (via terminal or cPanel Terminal)
2. **Check actual error**:
   ```bash
   tail -100 public_html/storage/logs/laravel.log
   ```
3. **Verify database**:
   ```bash
   php artisan tinker
   > DB::table('business_settings')->first();
   ```
4. **Run migrations if needed**:
   ```bash
   php artisan migrate --force
   ```
5. **Clear cache**:
   ```bash
   php artisan config:cache
   php artisan view:clear
   ```

---

## Files to Review/Update

| File | Purpose |
|------|---------|
| `.env` | Database credentials (most common issue) |
| `app/Providers/AppServiceProvider.php` | Fix the empty catch block |
| `storage/logs/laravel.log` | Actual error messages |
| `config/database.php` | Database config |

---

## Prevention for Future

Create a pre-deployment checklist:
- [ ] Test on staging/subdomain first
- [ ] Verify all .env variables match live server
- [ ] Run `php artisan migrate:fresh` locally, then database dump to live
- [ ] Check `storage/logs` for errors
- [ ] Test critical user flows (browse, add to cart, checkout)

